
%% Step 1: Determine if you have single-point or multi-point emissions
clear all;
close all;

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables';

% Specify the folders to process
folders = {''};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    directory = fullfile(parentPath, folderName);
    
    % List all CSV files in the directory
    fileList = dir(fullfile(directory, '*.csv'));

    % Create a new directory to store split tables based on emission points
    outputDirectory = fullfile(parentPath, folderName, 'Emission Sources');
    if ~exist(outputDirectory, 'dir')
        mkdir(outputDirectory);
    end

% Loop through each file in the folder
for fileIdx = 1:numel(fileList)
    % Load the CSV file
    filePath = fullfile(directory, fileList(fileIdx).name);
    data = readtable(filePath);

    % Count the number of unique values in the 'EmissionPoint' column
    uniquePoints = unique(data.EmissionPoint);
    numUniquePoints = numel(uniquePoints);
    
    % Add a column containing the count of unique points
    data.CountUniquePoints = repmat(numUniquePoints, height(data), 1);

    % Create a new directory based on the number of emission points
    if numUniquePoints == 1
        outputSubDirectory = fullfile(outputDirectory, 'Single_Point');
    else
        outputSubDirectory = fullfile(outputDirectory, 'Multi_Point');
    end

    % Create the subdirectory if it doesn't exist
    if ~exist(outputSubDirectory, 'dir')
        mkdir(outputSubDirectory);
    end

    % Copy the file to the appropriate directory
    destinationFilePath = fullfile(outputSubDirectory, fileList(fileIdx).name);
    copyfile(filePath, destinationFilePath);
    
    % Write the updated data table to a new CSV file with the count column
    outputFileName = fullfile(outputSubDirectory, [fileList(fileIdx).name]);
    writetable(data, outputFileName);
end

    % Count the number of unique values in the 'EmissionPoint' column
    uniquePoints = unique(data.EmissionPoint);
    numUniquePoints = numel(uniquePoints);
    
    % Add a column containing the count of unique points
    data.CountUniquePoints = repmat(numUniquePoints, height(data), 1);

    % Create a new directory based on the number of emission points
    if numUniquePoints == 1
        outputSubDirectory = fullfile(outputDirectory, 'Single_Point');
    else
        outputSubDirectory = fullfile(outputDirectory, 'Multi_Point');
    end

    % Create the subdirectory if it doesn't exist
    if ~exist(outputSubDirectory, 'dir')
        mkdir(outputSubDirectory);
    end

    % Copy the file to the appropriate directory
    destinationFilePath = fullfile(outputSubDirectory, fileList(fileIdx).name);
    copyfile(filePath, destinationFilePath);
    
    % Write the updated data table to a new CSV file with the count column
    outputFileName = fullfile(outputSubDirectory, [fileList(fileIdx).name]);
    writetable(data, outputFileName);
end
disp('Step 1 Complete')

%% Focus on Single_Point Sources first

% Step 2: Split into 5, 10 & 15 minute tables
clear all;
close all;

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables';

% Specify the folders to process
folders = {''};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    path = fullfile(parentPath, folderName,'Emission Sources', 'Single_Point');
    
    % For every event, determine 5, 10, 15-minute mini-tables
    % Initialize cell array to store mini-tables from all files
    allMiniTables = {};

    % Get a list of all files in the directory
    fileList = dir(fullfile(path, '*.csv'));

% Loop through each CSV file
for i = 1:numel(fileList)
    filename = fullfile(path, fileList(i).name);

    % Read the CSV file
    data = readtable(filename);

    % Filter out data below 1900 ppb - instrument starting
    %data = data(data.x_CH4__ppm > 1.9, :);
    
    % Convert the 'Time' column to datetime format
    data.Time = datetime(data.Time);

    % Define time intervals (5, 10, 15,30 minutes)
    interval = minutes(30);

    % Compute the start and end times for each interval
    startTime = min(data.Time);
    endTime = max(data.Time);
    timeIntervals = startTime:interval:endTime;

    % Split the data into 5,10,15,30-minute mini-tables based on the 'Time' column
    miniTables = cell(1, numel(timeIntervals) - 1);

    for j = 1:numel(timeIntervals)-1
        startTimeInterval = timeIntervals(j);
        endTimeInterval = timeIntervals(j+1);

        % Filter the data for the current interval
        averaged_table = data(data.Time >= startTimeInterval & data.Time < endTimeInterval, :);

        averaged_table.Actual_Emission_kg_h = 0.001*60 * 16.04 * 1 * averaged_table.C1FlowAvg_slpm / (0.082057 * 298);

        % Store the mini-table in the cell array if it's 5 minutes long
        if height(averaged_table) >= 14400 % At least 2400 rows are needed for a 5-minute interval- 8 hz sampling, 10 (4800) & 15 (7200)
            miniTables{j} = averaged_table;
        else
            fprintf('Skipping mini-table %d: Less than 30 minutes long.\n', j);
        end
    end

    % Add mini-tables from the current file to the cell array storing all mini-tables
    allMiniTables = [allMiniTables, miniTables];
end


    % Create a new folder named "30 minutes" for each folder
    newFolderPath = fullfile(path, '30 minutes');
    if ~exist(newFolderPath, 'dir')
        mkdir(newFolderPath);
    end

    % Loop through each mini-table and calculate average Actual_Emission_kg_h
    for k = 1:numel(allMiniTables)
        miniTable = allMiniTables{k};

        % Check if the mini-table is valid (not empty)
        if ~isempty(miniTable)
            % Find unique rows based on the first column (EmissionPoint)
            [uniqueIndices, ~, uniqueValues] = unique(miniTable.EmissionPoint);

            % Calculate the sum of Actual_Emission_kg_h for each unique EmissionPoint
            Emissions = accumarray(uniqueValues, miniTable.Actual_Emission_kg_h, [], @(x) x(1));

            % Calculate the total sum of emissions across all unique emission points
            TotalEmissions = sum(Emissions);

            % Create a new column named 'TotalEmissions' with the total emissions for each row
            miniTable.TotalEmissions = repmat(TotalEmissions, size(miniTable, 1), 1);

            % Get the start and end times
            startTime = miniTable.Time(1);
            endTime = miniTable.Time(end);

            % Calculate the duration of the mini-table
            duration = endTime - startTime;

            % Check if the duration is approximately 15 minutes
            if duration >= minutes(29.5) && duration <= minutes(30.5)
                % Create file names using start and end times
                startTimeStr = datestr(startTime, 'yyyy-mm-dd HH-MM-SS');
                endTimeStr = datestr(endTime, 'yyyy-mm-dd HH-MM-SS');
                fileName = sprintf('%s to %s.csv', startTimeStr, endTimeStr);
                filePath = fullfile(newFolderPath, fileName);

                % Write the mini-table to a CSV file
                writetable(miniTable, filePath);
            else
                fprintf('Skipping mini-table %d: Duration is not 30 minutes.\n', k);
            end
        end
    end
end
disp('Step 2 Complete')
%% Step 3 calculate bearing
clear all;
close all;

% Define the function to calculate initial bearing
calculate_initial_bearing = @(lat1, lon1, lat2, lon2) mod(rad2deg(atan2(...
    sin(deg2rad(lon2 - lon1)) .* cos(deg2rad(lat2)), ...
    cos(deg2rad(lat1)) .* sin(deg2rad(lat2)) - sin(deg2rad(lat1)) .* cos(deg2rad(lat2)) .* cos(deg2rad(lon2 - lon1)))) ...
    + 360, 360);

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables';

% Specify the folders to process
folders = {'30 minutes'};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    path = fullfile(parentPath, 'Emission Sources', 'Single_Point', folderName);
   
    % Get a list of all files in the directory
    fileList = dir(fullfile(path, '*.csv'));
    
    % Loop through each CSV file
    for i = 1:numel(fileList)
        filename = fullfile(path, fileList(i).name);

        % Read the CSV file
        data = readtable(filename);
        
        % Calculate initial bearing for each row and update directly in degrees
        data.bearing = arrayfun(@(lat1, lon1, lat2, lon2) calculate_initial_bearing(lat1, lon1, lat2, lon2), ...
                                data.Lat, data.Lon, data.MGGA_lat_N, data.MGGA_long_W);
        
        % Write the updated table
        writetable(data, filename);
    end
end

disp('Step 3 Complete');




%% Step 4: Filter data to within 10, 20, and 45 degrees- based of average WD
clear all;
close all;

% Load non-release data
non_release_data = readtable("\\csunas01.colostate.edu\research\metec\user\mbua\ADED\Eddy Covariance\8_Hz\non-release data.csv");
non_release_data = non_release_data(non_release_data.x_CH4__ppm > 0,:);
non_release_data.Time = datetime(non_release_data.Time, 'InputFormat', 'yyyy-MM-dd HH:mm:ss');

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables';

% Specify the folders to process
folders = {'5 minutes','15 minutes','30 minutes'};

% Define the angle thresholds
angleThresholds = [10 20 45];

% Loop through each angle threshold
for k = 1:numel(angleThresholds)
    angle = angleThresholds(k);
    
    % Loop through each folder
    for f = 1:numel(folders)
        folderName = folders{f};
        path = fullfile(parentPath, 'Emission Sources', 'Multi_Point', 'Emission Point',folderName);
       
        % Create a new directory for quantified files if it doesn't exist
        quantifiedPath = fullfile(path, [num2str(angle) ' degrees']);
        if ~exist(quantifiedPath, 'dir')
            mkdir(quantifiedPath);
        end
        
        % Get a list of all files in the directory
        fileList = dir(fullfile(path, '*.csv'));
        
        % Loop through each CSV file
        for i = 1:numel(fileList)
            filename = fullfile(path, fileList(i).name);

            % Read the CSV file
            data = readtable(filename);


     
                fileTime = datetime(data.Time(1), 'InputFormat', 'yyyy-MM-dd HH:mm:ss');
                idx = find(non_release_data.Time <= fileTime, 1, 'last');
                
                if ~isempty(idx)
                    startTime = non_release_data.Time(idx) - minutes(5);
                    timeMask = non_release_data.Time >= startTime & non_release_data.Time <= non_release_data.Time(idx);
                    filteredData = non_release_data(timeMask, :);
                    
                    if ~isempty(filteredData)
                        percentile_5th = prctile(filteredData.x_CH4_d_ppm, 5);
                        low_values = filteredData.x_CH4_d_ppm(filteredData.x_CH4_d_ppm <= percentile_5th);
                        Cb = mean(low_values);
                    else
                        Cb = NaN;
                    end
                    
                    mean_CH4_ppm = mean(data.x_CH4_d_ppm);
                    data.Cb = repmat(Cb, height(data), 1);
                    data.mean_CH4_ppm = repmat(mean_CH4_ppm, height(data), 1);
                    
                    if Cb > mean_CH4_ppm
                        data.Cb = repmat(min(data.x_CH4_d_ppm), height(data), 1);
                    end
                    
                   data.x_CH4_d_ppm_enh = data.x_CH4_d_ppm - data.Cb;
            % Circular averaging of wind direction (WD)
            sinSum = sum(sind(data.WD), 'omitnan'); % Sum of sines
            cosSum = sum(cosd(data.WD), 'omitnan'); % Sum of cosines
            avgWD = atan2d(sinSum, cosSum); % Circular mean in degrees
            if avgWD < 0
                avgWD = avgWD + 360; % Ensure avgWD is in [0, 360)
            end
            
            % Calculate the difference between data.downwind and avgWD
            data.avg_WD = repmat(avgWD, height(data),1);
            data.downwind = mod(data.bearing - 180, 360);
            data.diff = abs(data.downwind - avgWD);
            data.diff = min(data.diff, 360 - data.diff); % Handle angular wrap-around

            
            % Check if any difference is within the threshold
            if any(data.diff <= angle)
                % Write the entire table to a new CSV file in the quantifiedPath
                [~, fileNameWithoutExtension, extension] = fileparts(filename);
                newFilename = fullfile(quantifiedPath, [fileNameWithoutExtension extension]);
                writetable(data, newFilename);
            end
        end
    end
end
end
disp('Step 4 Complete');

%% Step 5: Move MRMP emissions to a different folder
clear all;
close all;
clc;
% Define the base directory path
basePath = '\\csunas01.colostate.edu\research\metec\user\mbua\ADED\Eddy Covariance\8_Hz\Event Tables\Emission Sources\Multi_Point\Emission Point';

% Define the subfolder structures to loop through
minuteFolders = {'5 minutes', '15 minutes','30 minutes'};
degreeFolders = {'10 degrees', '20 degrees', '45 degrees'};

% Loop through each minute folder
for minIdx = 1:length(minuteFolders)
    % Loop through each degree folder
    for degIdx = 1:length(degreeFolders)
        % Construct the full source path
        sourcePath = fullfile(basePath, minuteFolders{minIdx}, degreeFolders{degIdx});
        
        % Skip if this folder doesn't exist
        if ~exist(sourcePath, 'dir')
            fprintf('Folder does not exist: %s\n', sourcePath);
            continue;
        end
        
        % Create the 'Multi Emissions' folder if it doesn't exist
        destPath = fullfile(sourcePath, 'Multi Emissions');
        if ~exist(destPath, 'dir')
            mkdir(destPath);
        end

        % Get all CSV files in the directory
        fileList = dir(fullfile(sourcePath, '*.csv'));
        fileNames = {fileList.name};
        
        % Skip if no files found
        if isempty(fileNames)
            fprintf('No CSV files found in: %s\n', sourcePath);
            continue;
        end

        % Extract timestamps from filenames
        timestamps = cell(size(fileNames));
        for i = 1:length(fileNames)
            % Extract the timestamp part (everything before the last underscore)
            nameParts = strsplit(fileNames{i}, '_');
            timestamps{i} = strjoin(nameParts(1:end-1), '_');
        end

        % Find unique timestamps and their counts
        [uniqueTimestamps, ~, idx] = unique(timestamps);
        counts = accumarray(idx, 1);

        % Find indices of files with duplicate timestamps
        duplicateIndices = find(counts(idx) > 1);

        % Move files with duplicate timestamps
        if ~isempty(duplicateIndices)
            fprintf('\nProcessing folder: %s\n', sourcePath);
            fprintf('Found %d files with duplicate timestamps. Moving them...\n', length(duplicateIndices));
            
            for i = 1:length(duplicateIndices)
                srcFile = fullfile(sourcePath, fileNames{duplicateIndices(i)});
                destFile = fullfile(destPath, fileNames{duplicateIndices(i)});
                
                % Check if file exists in source before moving
                if exist(srcFile, 'file')
                    movefile(srcFile, destFile);
                    fprintf('Moved: %s\n', fileNames{duplicateIndices(i)});
                end
            end
            
            fprintf('Done moving files with duplicate timestamps.\n');
        else
            fprintf('\nProcessing folder: %s\n', sourcePath);
            fprintf('No files with duplicate timestamps found.\n');
        end
    end
end

fprintf('\nAll folders processed.\n');
